import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Code, Palette, Database, Calendar, CheckCircle, FileText, Users, Award, Zap, Clock, BookOpen } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedCourse, setSelectedCourse] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const courses = [
    { 
      id: 'algorithms', 
      name: 'Основы алгоритмизации и программирования', 
      icon: Code,
      color: '#2563EB',
      description: 'Изучите основы программирования, алгоритмы и структуры данных. Начните свой путь в IT с нуля.',
      duration: '120 часов',
      level: 'Начальный',
      students: '2500+',
      rating: '4.9'
    },
    { 
      id: 'web-design', 
      name: 'Основы веб-дизайна', 
      icon: Palette,
      color: '#7C3AED',
      description: 'Создавайте красивые и функциональные веб-интерфейсы. Изучите UI/UX дизайн и современные инструменты.',
      duration: '100 часов',
      level: 'Начальный',
      students: '1800+',
      rating: '4.8'
    },
    { 
      id: 'databases', 
      name: 'Основы проектирования баз данных', 
      icon: Database,
      color: '#10B981',
      description: 'Освойте проектирование и работу с базами данных. SQL, нормализация, оптимизация запросов.',
      duration: '140 часов',
      level: 'Средний',
      students: '1200+',
      rating: '4.9'
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedCourse || !startDate) {
      alert('Пожалуйста, выберите курс и дату начала обучения')
      return
    }

    const saved = localStorage.getItem('courseApplications')
    const applications = saved ? JSON.parse(saved) : []
    
    const courseData = courses.find(c => c.id === selectedCourse)
    
    const newApplication = {
      id: Date.now(),
      courseName: courseData.name,
      startDate: startDate,
      status: 'active',
      review: null
    }
    
    applications.push(newApplication)
    localStorage.setItem('courseApplications', JSON.stringify(applications))
    
    setIsSubmitted(true)
    
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  const selectedCourseData = courses.find(c => c.id === selectedCourse)

  return (
    <div className="enroll-split">
      <div className="split-layout-enroll">
        <aside className="sidebar-enroll">
          <div className="sidebar-enroll-content">
            <div className="sidebar-enroll-header">
              <div className="enroll-icon-wrapper">
                <BookOpen size={32} />
              </div>
              <h2 className="sidebar-enroll-title">Запись на курс</h2>
              <p className="sidebar-enroll-subtitle">
                Выберите курс и дату начала обучения
              </p>
            </div>

            <div className="enroll-steps">
              <div className={`enroll-step ${selectedCourse ? 'active completed' : ''}`} id="step-1">
                <div className="step-circle">
                  <span>1</span>
                </div>
                <div className="step-info">
                  <div className="step-label">Выбор курса</div>
                  <div className="step-status" id="step-1-status">
                    {selectedCourse ? 'Выбрано' : 'Не выбрано'}
                  </div>
                </div>
              </div>
              <div className={`enroll-step ${startDate ? 'active completed' : selectedCourse ? 'active' : ''}`} id="step-2">
                <div className="step-circle">
                  <span>2</span>
                </div>
                <div className="step-info">
                  <div className="step-label">Дата начала</div>
                  <div className="step-status" id="step-2-status">
                    {startDate ? 'Выбрано' : 'Не выбрано'}
                  </div>
                </div>
              </div>
            </div>

            {selectedCourseData && (() => {
              const PreviewIcon = selectedCourseData.icon
              return (
                <div className="selected-course-preview">
                  <div className="preview-header">
                    <div className="preview-icon" style={{ background: `${selectedCourseData.color}15` }}>
                      <PreviewIcon size={24} color={selectedCourseData.color} />
                    </div>
                    <h3 className="preview-title">{selectedCourseData.name}</h3>
                  </div>
                  <div className="preview-meta">
                    <div className="preview-meta-item">
                      <Clock size={16} />
                      <span>{selectedCourseData.duration}</span>
                    </div>
                    <div className="preview-meta-item">
                      <Users size={16} />
                      <span>{selectedCourseData.students}</span>
                    </div>
                  </div>
                </div>
              )
            })()}
          </div>
        </aside>

        <main className="main-panel-enroll">
          <div className="container">
            <section className="enroll-header-split">
              <h1 className="enroll-title-split">Выберите курс</h1>
              <p className="enroll-description-split">
                Все курсы включают практические задания, обратную связь от преподавателей и сертификат по завершении
              </p>
            </section>

            <section className="enroll-form-split">
              <form className="form-split" onSubmit={handleSubmit}>
                <div className="courses-selection-split">
                  {courses.map((course) => {
                    const Icon = course.icon
                    return (
                      <button
                        key={course.id}
                        type="button"
                        className={`course-card-split ${selectedCourse === course.id ? 'selected' : ''}`}
                        onClick={() => setSelectedCourse(course.id)}
                        style={{ '--course-color': course.color }}
                      >
                        <div className="course-card-header-split">
                          <div className="course-icon-split" style={{ background: `${course.color}15`, color: course.color }}>
                            <Icon size={32} />
                          </div>
                        </div>
                        <h3 className="course-name-split">{course.name}</h3>
                        <p className="course-desc-split">{course.description}</p>
                        <div className="course-stats-split">
                          <div className="course-stat-split">
                            <Clock size={16} />
                            <span>{course.duration}</span>
                          </div>
                          <div className="course-stat-split">
                            <Users size={16} />
                            <span>{course.students}</span>
                          </div>
                          <div className="course-stat-split">
                            <Award size={16} />
                            <span>{course.rating}</span>
                          </div>
                        </div>
                        <div className="course-level-split">
                          <span>{course.level}</span>
                        </div>
                      </button>
                    )
                  })}
                </div>

                {selectedCourse && (
                  <div className="date-section-split">
                    <div className="date-label-split">
                      <Calendar size={20} />
                      <span>Дата начала занятий</span>
                    </div>
                    <input
                      type="date"
                      className="date-input-split"
                      value={startDate}
                      onChange={(e) => setStartDate(e.target.value)}
                      min={minDateStr}
                      max={maxDateStr}
                      required
                    />
                  </div>
                )}

                <button 
                  type="submit" 
                  className="submit-btn-split" 
                  disabled={!selectedCourse || !startDate || isSubmitted}
                >
                  <span>Подать заявку</span>
                </button>

                {isSubmitted && (
                  <div className="success-message-split">
                    <CheckCircle size={32} />
                    <span>Заявка успешно отправлена!</span>
                  </div>
                )}
              </form>
            </section>

            <ImageSlider />

            <section className="benefits-split">
              <h2 className="benefits-title-split">Преимущества обучения</h2>
              <div className="benefits-grid-split">
                <div className="benefit-item-split">
                  <div className="benefit-icon-split">
                    <FileText size={28} />
                  </div>
                  <h3 className="benefit-title-split">Практические задания</h3>
                  <p className="benefit-text-split">Реальные проекты и кейсы из индустрии</p>
                </div>
                <div className="benefit-item-split">
                  <div className="benefit-icon-split">
                    <Users size={28} />
                  </div>
                  <h3 className="benefit-title-split">Обратная связь</h3>
                  <p className="benefit-text-split">Персональные консультации с преподавателями</p>
                </div>
                <div className="benefit-item-split">
                  <div className="benefit-icon-split">
                    <Award size={28} />
                  </div>
                  <h3 className="benefit-title-split">Сертификат</h3>
                  <p className="benefit-text-split">Официальный документ по завершении курса</p>
                </div>
                <div className="benefit-item-split">
                  <div className="benefit-icon-split">
                    <Zap size={28} />
                  </div>
                  <h3 className="benefit-title-split">Карьерная поддержка</h3>
                  <p className="benefit-text-split">Помощь в трудоустройстве после обучения</p>
                </div>
              </div>
            </section>
          </div>
        </main>
      </div>
    </div>
  )
}

export default Enroll

